-- Thanks for the tutorial of A Star : https://www.redblobgames.com/pathfinding/a-star/introduction.html
-- And a very nice article： https://www.gamedev.net/reference/articles/article2003.asp
-- 他的中文译版： https://blog.csdn.net/kenkao/article/details/5476392
-- And the Coordinate system of trailblazer mod： https://steamcommunity.com/sharedfiles/filedetails/?id=810372558
-- Written by 川小胖
-- Bi-directional Pathfinding Search
---------------------
-- For DebugPrint  --
---------------------
local DEBUG_MODE = false

local function DebugPrint(...)
	return DEBUG_MODE and print(...)
end
---------------------
----- Variables -----
---------------------

ASTAR_SPEED_FASTER = 1
ASTAR_SPEED_NORMAL = 0
ASTAR_SPEED_SLOWER = -1

ASTAR_COSTMULTI = {
	GROUND_SPEED = {SMALL = -0.5, NORMAL = 0, LARGE = 1}, 
	ANGLE_CHANGE = {NORMAL = 0.5, LARGE = 1},
}

------------------------- CONVERT FUNCTIONS ------------------------------
-- 返回一个基于玩家位置为原点的2D相对位置坐标，单位长度为PATH_NODE_DIST
-- Makes a 2int coordinate
-- @param x : x val of coordinate
-- @param y : y val of coodinate
-- @return  : 2int coordinate table:
--       .x : x val of coordinate
--       .y : y val of coordinate
--       .f_score : g_score + h_score
local makeCoord = function(x, y, f_score)
	return 	{
				x = x,
				y = y,
				f_score = f_score or 0
			}
end

-- 基于玩家原点的相对坐标转换为世界中的绝对位置
-- Converts a 2int coordinate to Vector3
-- @param origin : The origin of the coordinate system
-- @param coord  : coordinate to convert
-- @return       : the Vector3 in world space corresponding to the given coordinate
local coordToPoint = function(origin, coord, node_dist)
	return Vector3	(
						origin.x + (coord.x * node_dist),
						0,
						origin.z + (coord.y * node_dist)
					)
end

-- 世界位置 转换为 Step类型 （元素分别为y，x，z 的表结构）
-- Vector3 --> {y,x,z}
local pointToStep = function(point)
	return {y = point.y, x = point.x, z = point.z}
end

-- Step类型 转为 世界位置
-- {y,x,z} --> Vector3
local stepToPoint = function(step)
	return Vector3(step.x, step.y, step.z)
end
-------------------------------------------------------------------------
-- see https://www.gamedev.net/reference/articles/article2003.asp -- chapter"Notes on Implementation" -- Point 7


local BinaryHeap = require("utils/binaryheap")

----------------------------- COST FUNCTIONS -------------------------------
--- COST FUNCTIONS
-- 计算损失值，哈夫曼距离主要用于4方向，对角线距离用于8方向
-- calc the cost of G scores and F scores , F = G + H
-- when we consider four directions ,better to use Manhattan distance
-- when we consider eight directions ,better to use Diagnol distance
local CalcDistCost = function(p1, p2)
	-- Manhattan distance
	--return math.abs(p1.x - p2.x) + math.abs(p1.z - p2.z)
	
	-- Diagnol distance 
	local dx = math.abs(p1.x - p2.x)
	local dz = math.abs(p1.z - p2.z)
	local min_xz = math.min(dx,dz)
	return dx + dz - 0.5 * min_xz  --0.5 means is approximately equal to (2- sqrt(2))
end

-- 计算地面速度乘数，通过减小在卵石路上的G值可以优先考虑卵石路上的点，达到跟随卵石路的效果
-- 同样通过增大蜘蛛网上的G值可以尽量避免走到蜘蛛网上，（Webber则是加速）
local CalcGroundSpeedMulti = function(point, groundcaps)
    --NO GROUNDCAPS SETTINGS
    if groundcaps == nil then return ASTAR_COSTMULTI.GROUND_SPEED.NORMAL end

	--FLOOD (island adventure Mod)
	-- if groundcaps.speed_on_flood ~= nil then
	-- 	local is_onflood = TheWorld.components.flooding ~= nil and TheWorld.components.flooding.OnFlood and
	-- 							TheWorld.components.flooding:OnFlood(point.x, 0, point.z)
	-- 	if is_onflood then
    --         if groundcaps.speed_on_flood < ASTAR_SPEED_NORMAL then
    --             return ASTAR_COSTMULTI.GROUND_SPEED.LARGE
    --         elseif groundcaps.speed_on_flood > ASTAR_SPEED_NORMAL then
    --             return ASTAR_COSTMULTI.GROUND_SPEED.SMALL
    --         end
	-- 	end
	-- end

    --CREEP (actually it should check the walkable between points instead of walkable of point, but it should be okay since creep always fill a tile)
    if groundcaps.speed_on_creep ~= nil then
        local is_oncreep = TheWorld.GroundCreep:OnCreep(point.x, 0, point.z) -- CheckWalkableFromPoint(curpoint, point, {ignorewalls = true, allowocean = true, ignorecreep = false})
        -- should walk around the spidernet evenif it's on road or faster tiles
        if is_oncreep then
            if groundcaps.speed_on_creep < ASTAR_SPEED_NORMAL then
                return ASTAR_COSTMULTI.GROUND_SPEED.LARGE
            elseif groundcaps.speed_on_creep > ASTAR_SPEED_NORMAL then
                return ASTAR_COSTMULTI.GROUND_SPEED.SMALL
            end
        end
    end

    --ROAD
	local current_ground_tile = TheWorld.Map:GetTileAtPoint(point.x, 0, point.z)
    if groundcaps.speed_on_road ~= nil then
        local is_onroad = RoadManager ~= nil and RoadManager:IsOnRoad(point.x, 0, point.z) or
                            --current_ground_tile == WORLD_TILES.ROAD
							GROUND_ROADWAYS ~= nil and GROUND_ROADWAYS[current_ground_tile]
        if is_onroad then
            -- nobody will walk slower on road but just in case
            if groundcaps.speed_on_road < ASTAR_SPEED_NORMAL then
                return ASTAR_COSTMULTI.GROUND_SPEED.LARGE
            elseif groundcaps.speed_on_road > ASTAR_SPEED_NORMAL then
                return ASTAR_COSTMULTI.GROUND_SPEED.SMALL
            end
        end
    end
    --FASTER TILES
    if groundcaps.faster_on_tiles ~= nil then
        if groundcaps.faster_on_tiles[tostring(current_ground_tile)] then
            return ASTAR_COSTMULTI.GROUND_SPEED.SMALL
        end
    end
    return ASTAR_COSTMULTI.GROUND_SPEED.NORMAL
end

-- 计算角度变化乘数, 与上类似，但是按角度分段
local CalcAngleMulti = function(nextCoord_direct, currentCoord_direct)
	if nextCoord_direct == nil or currentCoord_direct == nil then return ASTAR_COSTMULTI.ANGLE_CHANGE.NORMAL end

	local nextCoord_angle = VecUtil_GetAngleInRads(nextCoord_direct.x, nextCoord_direct.z)
	local currCoord_angle = VecUtil_GetAngleInRads(currentCoord_direct.x, currentCoord_direct.z)
	local diff_angle = math.abs(nextCoord_angle - currCoord_angle)
	if diff_angle > PI then
		diff_angle = PI*2 - diff_angle
	end
	return Lerp(ASTAR_COSTMULTI.ANGLE_CHANGE.NORMAL, ASTAR_COSTMULTI.ANGLE_CHANGE.LARGE, diff_angle/PI)
end

local CalcTotalMulti = function(curr_pt, next_pt, old_dir, new_dir, groundcaps, boatRadius)
	local groundSpeedMulti = ASTAR_COSTMULTI.GROUND_SPEED.NORMAL
	local angleChangeMulti = ASTAR_COSTMULTI.ANGLE_CHANGE.NORMAL
	if true then
		groundSpeedMulti = CalcGroundSpeedMulti(next_pt, groundcaps)
	end
	if groundSpeedMulti == ASTAR_COSTMULTI.GROUND_SPEED.SMALL then  -- 
		angleChangeMulti = CalcAngleMulti(old_dir, new_dir)
	end
	return 1 + groundSpeedMulti + angleChangeMulti
end
------------------------------- WALKABLE CHECK FUNCTIONS----------------------------------

-- 陆地和浅海交界处，浅海地皮会伸出 1/4区域 的陆地，overhang区因为算浅海地皮，和陆地没有LOS
-- overhang area means the extend area that near the shallow ocean, it's belong as shallow ocean tile
local IsOverhangAtPoint = function(x, y, z)
	if type(x) == "table" and x.IsVector3 ~= nil and x:IsVector3() then
		x, y, z = x:Get()
	end
	return not TheWorld.Map:IsAboveGroundAtPoint(x, y, z) and TheWorld.Map:IsVisualGroundAtPoint(x, y, z)
end

-- 单点可走
-- walkable of one point(on ocean land tile, creep, walls, flood)
local function IsWalkablePoint(point, pathcaps)
	pathcaps = pathcaps or {allowocean = false, ignoreLand = false, ignorewalls = false, ignorecreep = false}

	-- check ocean and Land
	if not pathcaps.allowocean or pathcaps.ignoreLand then
		local is_onland = TheWorld.Map:IsVisualGroundAtPoint(point.x, 0, point.z)
		if not pathcaps.allowocean and not is_onland then -- not allow ocean but actually on ocean
			return false
		end
		if pathcaps.ignoreLand and is_onland then -- not allow land but actually on land
			return false
		end
	end
	-- check the creep
	if not pathcaps.ignorecreep then
		local is_oncreep = TheWorld.GroundCreep:OnCreep(point.x, 0, point.z)
		if is_oncreep then
			return false
		end
	end
	-- DEPRECATED, see bypass prefab defs "network_flood"
	-- -- check the flood
	-- if not pathcaps.ignoreflood then
	-- 	local is_onflood = TheWorld.components.flooding ~= nil and TheWorld.components.flooding.OnFlood and
	-- 							TheWorld.components.flooding:OnFlood(point.x, 0, point.z)
	-- 	if is_onflood then
	-- 		return false
	-- 	end
	-- end

	-- check the walls
	if not pathcaps.ignorewalls then
		local has_wall = TheWorld.Pathfinder:HasWall(point.x, 0, point.z)
		if has_wall then
			return false
		end
	end


	return true
end

-- 两点之间采样检测连通性
-- it's so expensive to check tiles walkable between points with long distance
local function SamplingCheckWalkable(pos, target_pos, pathcaps, sampling_dist)
	local vec, len = (target_pos - pos):GetNormalizedAndLength()
	local pathcaps = pathcaps or {}
	local sampling_dist = sampling_dist or 1
	if len > sampling_dist then -- the points dist smaller than sampling_dist, just skip
		local i = sampling_dist
		while(i < len) do
			local checkPos = pos + vec*i
			if not IsWalkablePoint(checkPos, pathcaps) then
				return false
			end
			i = i + sampling_dist
		end
	end
	return true
end

-- 检测两点连通性， 先用C接口检测，再手动检测某些可能连通的情况
-- check the LOS and potiental LOS
local function CheckWalkableFromPoint(pos, target_pos, pathcaps, boat_radius, without_check_potiental_walkable)
	local pathcaps = pathcaps or {ignorewalls = false,  ignorecreep = false, ignoreLand = false, allowocean = false, }
	-- precheck
	if pos == target_pos then
		return true
	end

	--------------------------------------------
	--[[ check flood individually for island adventure world]]
	-- DEPRECATED, see bypass prefab defs "network_flood"
	--------------------------------------------
	-- if not pathcaps.ignoreflood then
	-- 	local check_flood_pathcaps = {ignorewalls = true, ignorecreep = true, allowocean = true}
	-- 	local is_sampledpos_no_flood = SamplingCheckWalkable(pos, target_pos, check_flood_pathcaps)
	-- 	if not is_sampledpos_no_flood then
	-- 		return false, "flood blocked"
	-- 	end
	-- end

	--------------------------------------------
	--[[ check Collision with Land if we are moving a boat ]]
	--------------------------------------------
	-- LOS check if has land blocked on two sides of boat trajectory
	if type(boat_radius) == "number" then
		local vec = (target_pos - pos):GetNormalized()
		local normal_vecs = {Vector3(-vec.z, 0, vec.x), Vector3(vec.z, 0, -vec.x)} --法线向量
		for k, v in ipairs(normal_vecs) do
			local start_pos, end_pos = pos + v * boat_radius, target_pos + v * boat_radius
			if not TheWorld.Pathfinder:IsClear(
											start_pos.x, 0, start_pos.z,
											end_pos.x, 0, end_pos.z, 
											{allowocean = true, ignoreLand = true, ignorewalls = true, ignorecreep = true} -- only check tiles connection
										) then
				return false, "land blocks boat trajectory"
			end
		end
	end
	
	--------------------------------------------
	--[[ check Distance from Lava if we are in a valcano world. eg:Island Advent Mod ]]
	--------------------------------------------
	if WORLD_TILES and WORLD_TILES.VOLCANO_LAVA then
		local dist = 15
		local vec = (target_pos - pos):GetNormalized()
		local normal_vecs = {Vector3(-vec.z, 0, vec.x), Vector3(vec.z, 0, -vec.x)} --法线向量
		for k, v in ipairs(normal_vecs) do
			local start_pos, end_pos = pos + v * dist, target_pos + v * dist
			if not TheWorld.Pathfinder:IsClear(
											start_pos.x, 0, start_pos.z,
											end_pos.x, 0, end_pos.z, 
											{allowocean = false, ignoreLand = false, ignorewalls = true, ignorecreep = true} -- only check tiles connection
										) then
				return false, "too close to the lava"
			end
		end
	end

	--------------------------------------------
	--[[ C side interface to check LOS ]]
	--------------------------------------------
	-- the C side pathfinder result
	-- this interface don't check the flood, diagonal tile connection points ,overhang points and fake ocean tiles area points
	local hasLOS = TheWorld.Pathfinder:IsClear(
			pos.x, pos.y, pos.z,
			target_pos.x, target_pos.y, target_pos.z,
			pathcaps
	)
	if hasLOS == true then
		return true
	end

	if without_check_potiental_walkable then
		return false, "IsClear return false"
	end

	-- when reach here, IsClear return false
	-- we are going to check diagonal tile connection points and overhang points and the points at void of Cave
	
	-- ABOUT POINTS ON ONLY Diagonal TILES CONNECTIONS:
	-- https://forums.kleientertainment.com/forums/topic/147232-the-issue-about-pathfinderisclear-on-tile-connection-in-diagonal-direction/
	-- when the points are in diagonal tile connection,IsClear(pathcaps={allowocean=false}) return false
	-- they can be actually walk through,so we should take account of it

	-- ABOUT POINTS ON OVERHANGE AREA:
	-- overhang area means the extra 1/4 TILE_SCALE visual ground area generated because of two different tiles connection
	-- IsClear(pathcaps={allowocean=false}) interface return false if one of points is at overhang area(allowocean = false)
	
	-- ABOUT POINT AT VOID OF CAVE:
	-- IsClear(pathcaps={allowocean=true}) return false when one of the points is at outbound tiles(fake ocean tiles) in the no ocean world (eg: Cave)
	-- to handle this we should take the points on outbound tiles in to account 
	--------------------------------------------
	--[[ check points with Diagonal Direction or At Overhang ]]
	--------------------------------------------
	if not pathcaps.allowocean then -- only land allowed only should we check the diagonal direction points and overhang points 
		local is_overhang = IsOverhangAtPoint(pos.x, pos.y, pos.z) or IsOverhangAtPoint(target_pos.x, target_pos.y, target_pos.z)
		local tile_x1, tile_y1 = TheWorld.Map:GetTileCoordsAtPoint(pos.x, pos.y, pos.z)
		local tile_x2, tile_y2 = TheWorld.Map:GetTileCoordsAtPoint(target_pos.x, target_pos.y, target_pos.z)
		local abs_dir_x, abs_dir_y = math.abs(tile_x1 - tile_x2), math.abs(tile_y1 - tile_y2)
		local is_diagonal_dir = (abs_dir_x > 0 and abs_dir_y > 0) and (abs_dir_x == abs_dir_y)
		if not is_diagonal_dir and not is_overhang then
			return false, "not at diagonal direction or overhang"
		end
	end


	--------------------------------------------
	--[[ check walkable at points]]
	--------------------------------------------
	local is_points_both_walkable = IsWalkablePoint(pos, pathcaps) and IsWalkablePoint(target_pos, pathcaps)
	if not is_points_both_walkable then
		return false, "not both walkable points"
	end

	--------------------------------------------
	--[[ check walls and creep between points(only works in world with ocean)]]
	--------------------------------------------
	-- set allowocean to exclude the ocean tile factor
	-- it's unfit for world has not ocean(eg: the cave) to set allowocean flag in order to check walls individually, it always return no LOS

	local world_has_ocean = TheWorld.has_ocean
	local skip_allowocean_check = not world_has_ocean

	if not skip_allowocean_check and not (pathcaps.ignorewalls and pathcaps.ignorecreep) then
		-- set allowocean to check walls and creep
		local check_pathcaps = shallowcopy(pathcaps)
		check_pathcaps.allowocean = true

		local walkable_exclude_oceanlandlimits = TheWorld.Pathfinder:IsClear(
						pos.x, pos.y, pos.z,
						target_pos.x, target_pos.y, target_pos.z,
						check_pathcaps
				)
		if not walkable_exclude_oceanlandlimits then
			return false, "walls or creep blocked"
		end
	end
	-- Check Completion:
	--- Diagonal Direction Points(allowocean = false):
	---- FOREST WORLD|| walls:√ 		creep:√ 	flood:√ 	oceanlandlimits:×
	---- CAVE   WORLD|| walls:× 		creep:× 	flood:√ 	oceanlandlimits:×
	--- Overhang Area Points:(allowocean = false):
	---- FOREST WORLD|| walls:√ 		creep:√ 	flood:√ 	oceanlandlimits:×
	---- CAVE   WORLD|| walls:× 		creep:× 	flood:√ 	oceanlandlimits:×
	--- Void Tiles Points:(allowocean = true):
	---- CAVE   WORLD|| walls:× 		creep:× 	flood:√ 	oceanlandlimits:×
	--------------------------------------------
	--[[ check sampled pos walkable(oceanlandlimits, creep, walls, )]]
	--------------------------------------------
	-- it's so expensive to check sampled pos for points with long distance, only use it if no other choice

	--take some samples point and check if is on land,to handle the two points are both at one of overhang area in same tile
	--check some point to ensure the connection between tiles
	local check_pathcaps = shallowcopy(pathcaps)
	if world_has_ocean then
		check_pathcaps.ignorewalls = true
		check_pathcaps.ignorecreep = true
		-- check_pathcaps.ignoreflood = true
	else
		-- check_pathcaps.ignoreflood = true
	end
	local is_sampledpos_walkable = SamplingCheckWalkable(pos, target_pos, check_pathcaps)
	if not is_sampledpos_walkable then
		return false, "sampled pos not walkable"
	end

	return true
end

------------------------------------- PATH OPERATE FUNCTIONS ------------------------------------------
-- 构建路径，从末尾节点向前遍历camefrom链表，其中只有方向不同的点才会插入到路径中，可以简化路径
-- Constructs a path from a finishedPath
-- @param search 	   : The search you request, which contains the params of pathfinding 
-- @param finalCoord   : The finalCoord that close the dest,but not the dest
-- @return             : The same path, stored in native format
local makePath = function(search, finalCoord)
	-- Convert came_from to the path 
	-- Structure: table
	-- .steps
	--       .1.y = 0
	--       .1.x = <x value>
	--       .1.z = <z value>
	--       ...
	
	-- construct path part
	local path = { steps = { } }
	local origin = search.startPos  -- original point of coord

	table.insert(path.steps, pointToStep(search.original_startPos))

	local init_steps = #(path.steps) + 1 -- the position that start to insert into
	for direction, subdata in pairs(search.data) do
		local reversed = direction == "end_to_start" -- keep the sync with this : search.data = {start_to_end= {}, end_to_start = {}}
		local finalPoint = coordToPoint(origin, finalCoord, search.node_dist)
		--local lastDirection = finalPoint - coordToPoint(origin, subdata.endCoord, search.node_dist)
		local lastDirection = nil
		local currentCoord = finalCoord
		local lastCoord = currentCoord
		while currentCoord ~= nil do
			local currentDirection = subdata.direction_so_far[currentCoord.x] and subdata.direction_so_far[currentCoord.x][currentCoord.y] or nil
			-- In order to simplify the path, only the steps with different direction will be added
			if currentDirection ~= lastDirection then
				local worldPoint = coordToPoint(origin, currentCoord, search.node_dist)
				--local point = Vector3(worldVec:Get()) It's Wrong!
				-- Notice: the step is a table of {x,y,z}, not Vector3
				-- to keep the same with klei's pathfinding result format
				local step = pointToStep(worldPoint)
				if reversed then
					table.insert(path.steps, step)
				else
					table.insert(path.steps, init_steps , step)
				end
			end
			lastDirection = currentDirection
			lastCoord = currentCoord
			currentCoord = subdata.came_from[currentCoord.x] and subdata.came_from[currentCoord.x][currentCoord.y] or nil
		end
	end

	table.insert(path.steps, pointToStep(search.original_endPos))

	-- remove the search.startPos and search.endPos if they has LOS
	local stepnum = #path.steps
	if stepnum - 2 > 0 and CheckWalkableFromPoint(stepToPoint(path.steps[stepnum]), stepToPoint(path.steps[stepnum-2]), search.pathcaps, search.boatRadius) then
		table.remove(path.steps, stepnum-1)
		stepnum = stepnum - 1
	end
	if stepnum >= 3 and CheckWalkableFromPoint(stepToPoint(path.steps[1]), stepToPoint(path.steps[3]), search.pathcaps, search.boatRadius) then
		table.remove(path.steps, 2)
	end

	return path
end


-- 另一种平滑路径的方法，在构建完路径后跑一遍，每三个点两头hasLos，则中间点为非必要点
-- a method to smooth the path via run through the path and check LOS between points
-- remove the unneccessary points which has LOS ,except the points on the road
local smoothPath = function(path, pathcaps, groundcaps, boatRadius)
	-- smooth path part
	-- ie: {0,0}, {0,2}, {3,2} -> {0,0}, {3,2} (given LOS)
	local index = 2
	local check_pathcaps = shallowcopy(pathcaps)
	-- reset the ignorecreep = false , to test whether the point can help me avoid the creep
	if groundcaps and groundcaps.speed_on_creep and groundcaps.speed_on_creep < 0 then
		check_pathcaps.ignorecreep = false
	end
	---- DEPRECATED, see bypass prefab defs "network_flood"
	-- -- reset the ignoreflood = false , to test whether the point can help me avoid the flood
	-- if groundcaps.speed_on_flood and groundcaps.speed_on_flood < 0 then
	-- 	check_pathcaps.ignoreflood = false
	-- end

	while(index < #(path.steps)) do
    
		-- Points to test
		local pre = path.steps[index-1]
		local post = path.steps[index+1]
		local cur = path.steps[index]

		local prePoint, curPoint, postPoint = stepToPoint(pre), stepToPoint(cur), stepToPoint(post)
		-- dont remove the points that on speedup_turf even if they're have LOS with previous point
		if CheckWalkableFromPoint(prePoint, postPoint, check_pathcaps, boatRadius) and -- Has LOS
			-- (boatRadius == nil or CalcLandDistMulti(prePoint, postPoint, boatRadius) ~= ASTAR_COSTMULTI.LAND_DIST.LARGE) and
			CalcGroundSpeedMulti(curPoint, groundcaps) ~= ASTAR_COSTMULTI.GROUND_SPEED.SMALL then -- not on faster tiles 
			table.remove(path.steps, index)
		else -- No LOS
			index = index + 1
		end
	end
	return path

end

-- 找自身和附近地皮的Walkable中心点
-- find the walkable center point  from self tile and nearby tiles
-- to avoid the situation when point is overhang that hasnoLOS with any other points
local function FindNearbyWalkableCenterPoint(point, pathcaps, check_los)
	local x, y, z = point:Get()
	local is_landtile = TheWorld.Map:IsVisualGroundAtPoint(x, y ,z)
	if not is_landtile then return nil end

	--local is_overhang = is_landtile and not TheWorld.Map:IsAboveGroundAtPoint(x, y, z)
	--local with_checklos = not is_overhang

	local resultCenterPoint = nil

	local currentTileCenterPoint = Vector3(TheWorld.Map:GetTileCenterPoint(x, y, z))
	-- consider current tile first
	if not check_los and IsWalkablePoint(currentTileCenterPoint, pathcaps) or CheckWalkableFromPoint(point, currentTileCenterPoint, pathcaps) then
		resultCenterPoint = currentTileCenterPoint
	else
		-- consider nearby tiles next
		local breakFlag = false
		for dx= -1, 1 ,1 do
			for dz = -1, 1, 1 do
				local nearbyTileCenterPoint = currentTileCenterPoint+Vector3(dx, 0, dz)*TILE_SCALE
				if not check_los and IsWalkablePoint(nearbyTileCenterPoint, pathcaps) or CheckWalkableFromPoint(point, nearbyTileCenterPoint, pathcaps) then
					resultCenterPoint = nearbyTileCenterPoint
					breakFlag = true
					break
				end
			end
			if breakFlag then
				break
			end
		end
	end

	if resultCenterPoint == nil then
		DebugPrint("no fitable point")
	end

	return resultCenterPoint or nil
end


-- 请求一个搜索，参数为起点，终点，路径设置（ignorecreep = true 指忽视即可以穿过蜘蛛网）和特殊地面设置（卵石路上和蛛网的速度变化，false则不考虑）
-- 搜索中包含了路径的信息，和用于寻路的一些初始变量
-- @param startPos : A Vector3 containing the starting position in world units
-- @param endPos   : A Vector3 continaing the ending position in world units
-- @param pathcaps : (Optional) the pathcaps to use for pathfinding
-- @param groundcaps: (Optional) whether movement speed get changed in road /in creep (spider-web)
-- @return         : A partial path object
--                 .path : If path is finished via LOS, this will be populated, otherwise nil
local requestSearch = function(startPos, endPos, pathcaps, groundcaps, boatRadius)
	
	----------------------
	-- Store parameters --
	----------------------
	local search = { }

	-- LOS parameter
	search.pathcaps = pathcaps and shallowcopy(pathcaps) or {}
	search.pathcaps.player = true

	-- better to set it in pathfollower
	-- i have set the penalty factor for creep and flood in CalcGroundSpeedMulti
	search.pathcaps.ignorecreep = true
	-- search.pathcaps.ignoreflood = true

	-- should consider to leave the gap from the shore to avoid the collision if we have the boat
	search.boatRadius = boatRadius
	-- pathfinding with allowocean is 8 to avoid get too many passable points in openlist
	search.node_dist = (search.pathcaps.allowocean and 2 or 1) * TILE_SCALE

	---- handle the case that one of points is on the tile which pathfinding not allow, just early quit
	--check obviously no way to avoid unnecessary resource waste
    if not(IsWalkablePoint(startPos, search.pathcaps) and IsWalkablePoint(endPos, search.pathcaps)) then -- obviously no path
		search.path = {} -- early quit and finish with no way
		return search
	end

	search.groundcaps = groundcaps and shallowcopy(groundcaps) or {speed_on_road = nil, speed_on_creep = nil, faster_on_tiles = {}}-- nil means not consider road and creep

	-- perserve the real start and end position
    search.original_startPos = Vector3(startPos:Get())
    search.original_endPos = Vector3(endPos:Get())

	-- 2023.8.10: merge it to FindNearbyWalkableCenterPoint without check_los
	------ handle the case that points has wall , try to override it as nearby no wall point, otherwise early quit
	---- override the endPos as searching the nearby point if dest has wall
	--if not search.pathcaps.ignorewalls and not is_point_without_wall(endPos) then
	--	local inv_dir = startPos - endPos
	--	local start_angle = VecUtil_GetAngleInDegrees(inv_dir.x, inv_dir.z)
	--	local fitable_override_point = SearchNearbyPointWithoutWall(endPos, start_angle, TILE_SCALE, 8)
	--	if fitable_override_point ~= nil then
	--		endPos = fitable_override_point
	--	else -- obviously no path
	--		search.path = {}
	--		return search
	--	end
	--end

    ---- try to standardization and handle the case that points are at overhang area
    -- there'll be some issue without standardization in some case ,such as when the points in diagonal tile connection
    -- in some other case it failed to standardization and maybe okay to pathfinding with origin point
	search.startPos = FindNearbyWalkableCenterPoint(startPos, pathcaps, true) or search.original_startPos
	search.endPos   = FindNearbyWalkableCenterPoint(endPos, pathcaps) or search.original_endPos
	search.startCoord = makeCoord(0,0)
	search.endCoord = makeCoord(
								math.floor((search.endPos.x - search.startPos.x) / search.node_dist),
								math.floor((search.endPos.z - search.startPos.z) / search.node_dist)
							)

	-------------------------
	-- Prepare Pathfinding --
	-------------------------

	-- search variable init
	search.data = {
		start_to_end = {startCoord = search.startCoord, endCoord = search.endCoord},
		end_to_start = {startCoord = search.endCoord, endCoord = search.startCoord}
	} --bi-directional search
	for direction, subdata in pairs(search.data) do
		subdata.openlist = BinaryHeap:new()	-- use binary heap for optimalize
		subdata.openlist:push(subdata.startCoord)

		subdata.closedlist = { }			-- 2 dim array, coord's x,y as index, bool as element

		subdata.g_score_so_far = { }		-- 2 dim array, coord's x,y as index, number as element
		subdata.g_score_so_far[subdata.startCoord.x] = { }
		subdata.g_score_so_far[subdata.startCoord.x][subdata.startCoord.y] = 0

		subdata.direction_so_far = { }	-- 2 dim array, coord's x,y as index, Vector3 as element
		subdata.direction_so_far[subdata.startCoord.x] = { }
		subdata.direction_so_far[subdata.startCoord.x][subdata.startCoord.y] = Vector3(0,0,0)

		subdata.came_from = { }			-- 2 dim array, coord's x,y as index, coord as element
	end
	-- search info init
	search.path = nil
	search.totalWorkDone = 0
	search.startTime = os.clock() --[[GetTime() ]]

	return search
end

-- 处理搜索，放在PeriodicTask或者OnUpdate，输入是每轮的最大处理量，
-- 如果找到返回true，结果放在search.path里
-- 如果运行到超过最大量则返回false，因此应该放在PeroidicTask或者OnUpdate里，反复继续上次的执行，这种少量分次运行应该能更好处理中止
-- @param search 		 : The search to finish (request one via requestSearch)
-- @param timePerRound   : The process time of point track per-round
-- @param maxTime        : The search will process util reach the max total process time
-- @return    			 : true if search is over (path == nil means not found path, path is valid means we found it), false if reach the workPerRound you set
local processSearch = function(search, timePerRound, maxTime)

	-- Path already found, return
	if search.path ~= nil then
		return true
	end
	
	-- Cache parameters
	local origin = search.startPos
	
	-- Paths processed this run
	local round_start_time = os.clock()

	-- Process until finished (no search.openlist remain or a path is found)
	while not search.data.start_to_end.openlist:isEmpty() and not search.data.end_to_start.openlist:isEmpty() do
		
		-- get the coord of min F value
		local currentCoordStart = search.data.start_to_end.openlist:pop()
		local currentCoordEnd = search.data.end_to_start.openlist:pop()
		search.data.start_to_end.currentCoord = currentCoordStart
		search.data.end_to_start.currentCoord = currentCoordEnd

		--for direction, subdata in pairs(search.data) do
		--	subdata.closedlist[subdata.currentCoord.x] = subdata.closedlist[subdata.currentCoord.x] or {}
		--	subdata.closedlist[subdata.currentCoord.x][subdata.currentCoord.y] = true
		--end

		local finalCoord = nil
		if search.data.end_to_start.closedlist[currentCoordStart.x] and search.data.end_to_start.closedlist[currentCoordStart.x][currentCoordStart.y] then
			finalCoord = currentCoordStart
		elseif search.data.start_to_end.closedlist[currentCoordEnd.x] and search.data.start_to_end.closedlist[currentCoordEnd.x][currentCoordEnd.y] then
			finalCoord = currentCoordEnd
		end
		---------------------------------
		-- Successed in Pathfinding !! --
		---------------------------------
		--pathfinding finish
		if finalCoord ~= nil then
			DebugPrint("[A STAR PATHFINDER] : " .. "start generate path！")
			search.path = makePath(search, finalCoord)

			-- FIX ME: the smooth part may delete the neccessary point to keep the dist from shore
			-- if search.boatRadius == nil then
				search.path = smoothPath(search.path, search.pathcaps, search.groundcaps, search.boatRadius)
				DebugPrint("[A STAR PATHFINDER] : " .. "smoothed path！")
			-- end

			-- update info
			search.endTime = os.clock() --[[GetTime() ]]
			search.costTime = search.endTime - search.startTime
			DebugPrint("[A STAR PATHFINDER] : " .. "finish pathfinding !, cost time: " .. search.costTime .. ",tracked points :" .. search.totalWorkDone )
			return true
		end

		for direction, subdata in pairs(search.data) do
			subdata.closedlist[subdata.currentCoord.x] = subdata.closedlist[subdata.currentCoord.x] or {}
			subdata.closedlist[subdata.currentCoord.x][subdata.currentCoord.y] = true

			local currentCoord = subdata.currentCoord
			local currentPoint = coordToPoint(origin, currentCoord, search.node_dist)

			-- Candidate coordinates, 8 directions
			local neighborCoordinates = {
											makeCoord(currentCoord.x    , currentCoord.y + 1),
											makeCoord(currentCoord.x + 1, currentCoord.y	),
											makeCoord(currentCoord.x 	, currentCoord.y - 1),
											makeCoord(currentCoord.x - 1, currentCoord.y 	),
											makeCoord(currentCoord.x + 1, currentCoord.y + 1),
											makeCoord(currentCoord.x - 1, currentCoord.y - 1),
											makeCoord(currentCoord.x + 1, currentCoord.y - 1),
											makeCoord(currentCoord.x - 1, currentCoord.y + 1)
										}
			-- Process Candidates
			for _, coordinate in ipairs(neighborCoordinates) do -- only two direction actually, just for less redundant code
				local nextCoord = coordinate
				local nextPoint = coordToPoint(origin, nextCoord, search.node_dist)

				if not (subdata.closedlist[nextCoord.x] and subdata.closedlist[nextCoord.x][nextCoord.y]) and -- not in closed list
					CheckWalkableFromPoint(currentPoint, nextPoint, search.pathcaps, search.boatRadius) then	-- walkable between two points
					-- Update G scores

					-- walk on the road first and on the creep last
					-- 					local is_onroad = search.groundcaps.speed_on_road and (RoadManager ~= nil and RoadManager:IsOnRoad(nextPoint.x, 0, nextPoint.z) or TheWorld.Map:GetTileAtPoint(nextPoint.x, 0, nextPoint.z) == WORLD_TILES.ROAD) or false
					-- 					local is_oncreep = search.groundcaps.speed_on_creep and (TheWorld.GroundCreep:OnCreep(nextPoint.x, 0, nextPoint.z)) or false
					-- 					local is_on_faster_tiles = search.groundcaps.faster_on_tiles and search.groundcaps.faster_on_tiles[tostring(TheWorld.Map:GetTileAtPoint(nextPoint.x, 0, nextPoint.z))]
					local old_direction = subdata.direction_so_far[currentCoord.x][currentCoord.y]
					local new_direction = (nextPoint - currentPoint):GetNormalized()

					local cost = CalcDistCost(currentPoint, nextPoint)
					local multi = CalcTotalMulti(currentPoint, nextPoint, old_direction, new_direction, search.groundcaps, search.boatRadius)
					local new_cost = cost * multi + subdata.g_score_so_far[currentCoord.x][currentCoord.y]

					subdata.g_score_so_far[nextCoord.x] = subdata.g_score_so_far[nextCoord.x] or {}
					subdata.direction_so_far[nextCoord.x] = subdata.direction_so_far[nextCoord.x] or {}
					subdata.came_from[nextCoord.x] = subdata.came_from[nextCoord.x] or {}

					if subdata.g_score_so_far[nextCoord.x][nextCoord.y] == nil or new_cost < subdata.g_score_so_far[nextCoord.x][nextCoord.y] then
						subdata.g_score_so_far[nextCoord.x][nextCoord.y] = new_cost
						subdata.direction_so_far[nextCoord.x][nextCoord.y] = new_direction
						nextCoord.f_score = new_cost + CalcDistCost(nextPoint, coordToPoint(origin, subdata.endCoord, search.node_dist))
						--DebugPrint("[A STAR PATHFINDER] : " .. "f: "..nextCoord.f_score)
						if not subdata.openlist:contains(nextCoord) then
							subdata.openlist:push(nextCoord)
						else
							subdata.openlist:update(nextCoord)
						end
						subdata.came_from[nextCoord.x][nextCoord.y] = currentCoord
						--DebugPrint("[A STAR PATHFINDER] : " .. string.format("(%d,%d)-->(%d,%d)",currentCoord.x,currentCoord.y,nextCoord.x,nextCoord.y))

						-- Update work done
						--workDone = workDone + 1
						search.totalWorkDone = search.totalWorkDone + 1
					end
				end
			end
		end



		-- Check work
		local time = os.clock()
		if time - round_start_time > timePerRound then
			--search.totalWorkDone = search.totalWorkDone + workDone
			-- if reach the max work , just give up
			if time - search.startTime < maxTime then
				--print(os.clock() - round_start_time)
				return false	-- another try in next round
			else
				DebugPrint("[A STAR PATHFINDER] : " .. "forcestop because max process time reached, we have tracked points:" .. search.totalWorkDone .. " cost time:" .. time - search.startTime)
				return true		-- too many tries, forcestop
			end
		end
	end
	
	----------------------------
	-- Fail in Pathfinding !! --
	----------------------------

	-- No path found ,it happens when you didn't set a max work and it will track all the map.
	if search.path == nil then
		DebugPrint("[A STAR PATHFINDER] : " .. "we have tracked all the map! no path found !")
		return true
	end
	
end

return
{
	requestSearch = requestSearch,
	processSearch = processSearch,
	-- extra functions for astarpathfinder
	CheckWalkableFromPoint = CheckWalkableFromPoint,
	IsWalkablePoint = IsWalkablePoint,
	CalcGroundSpeedMulti = CalcGroundSpeedMulti,
}